<?php

namespace App\Http\Controllers;

use App\Models\Nasheed;
use App\Http\Requests\StoreNasheedRequest;
use App\Http\Requests\UpdateNasheedRequest;
use App\Http\Requests\UpdateNasheedOrderRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class NasheedController extends Controller
{
    public function index(Request $request)
    {
        $query = Nasheed::query()->orderBy('order')->orderBy('created_at', 'desc');

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('singer', 'like', "%{$search}%");
            });
        }

        $nasheeds = $query->get();

        if ($request->ajax()) {
            return view('nasheeds.partials.list', compact('nasheeds'))->render();
        }

        return view('nasheeds.index', compact('nasheeds'));
    }

    public function create()
    {
        return view('nasheeds.create');
    }

    public function store(StoreNasheedRequest $request)
    {
        $validated = $request->validated();

        $audioPath = null;
        if ($request->hasFile('audio')) {
            $audioPath = $request->file('audio')->store('nasheeds/audio', 'public');
        }

        $order = Nasheed::max('order') + 1;

        Nasheed::create([
            'name' => $validated['name'],
            'singer' => $validated['singer'],
            'audio_path' => $audioPath,
            'lyrics' => $validated['lyrics'] ?? null,
            'order' => $order,
        ]);

        return redirect()->route('nasheeds.index')
            ->with('success', 'تم إضافة النشيد بنجاح');
    }

    public function edit(Nasheed $nasheed)
    {
        return view('nasheeds.edit', compact('nasheed'));
    }

    public function update(UpdateNasheedRequest $request, Nasheed $nasheed)
    {
        $validated = $request->validated();

        $audioPath = $nasheed->audio_path;

        if ($request->hasFile('audio')) {
            // Delete old file
            if ($audioPath) {
                Storage::disk('public')->delete($audioPath);
            }
            $audioPath = $request->file('audio')->store('nasheeds/audio', 'public');
        }

        $nasheed->update([
            'name' => $validated['name'],
            'singer' => $validated['singer'],
            'audio_path' => $audioPath,
            'lyrics' => $validated['lyrics'] ?? null,
        ]);

        return redirect()->route('nasheeds.index')
            ->with('success', 'تم تعديل النشيد بنجاح');
    }

    public function destroy(Nasheed $nasheed)
    {
        if ($nasheed->audio_path) {
            Storage::disk('public')->delete($nasheed->audio_path);
        }

        $nasheed->delete();

        return redirect()->route('nasheeds.index')
            ->with('success', 'تم حذف النشيد بنجاح');
    }

    /**
     * Update the order of nasheeds via AJAX.
     */
    public function updateOrder(UpdateNasheedOrderRequest $request)
    {
        $validated = $request->validated();

        foreach ($validated['ids'] as $index => $id) {
            Nasheed::where('id', $id)->update(['order' => $index + 1]);
        }

        return response()->json(['success' => true]);
    }
}
